/*
 *
 * This original maze generation program created
 *  entirely from scratch by Danny Brewer.
 * Program started on 4/20/1998.
 * Heavily revised and improved 3/1/2003.
 * Copyright 1998 Danny Brewer.  All rights reserved.
 * 
 * Copyright 2003 Danny Brewer
 * Anyone may run this code.
 * If you wish to modify or distribute this code, then
 *  you are granted a license to do so only under the terms
 *  of the Gnu Lesser General Public License.
 * See:  http://www.gnu.org/licenses/lgpl.html
 */


package nom.DannyBrewer.recreation.squareMaze;

import java.util.Random;


public class SquareMazeGenerator1 {
    protected static final boolean DEBUG = false;
    
    private Random random = new Random();
    //	private Random random = new Random( 2856 );	// Fixed outcome for debugging
    
    //	Both the solution path and the dead end paths
    //	 can be adjusted for how straight or twisty they are.
    //	Use 0.0 for maximum straightness,
    //	 use 1.0 for maximum twistyness.
    //
    private float twistFactorSolution = 0.4f;
    private float twistFactorDeadEnd = 0.4f;
    
    private SquareMaze maze = null;
    
    
    //----------------------------------------------------------------------
    //  Constructor
    //----------------------------------------------------------------------
    
    public SquareMazeGenerator1( SquareMaze maze ) {
        this.maze = maze;
    }
    
    
    //----------------------------------------------------------------------
    //  Public API
    //----------------------------------------------------------------------
    
    public void generateMaze() {
        formSolutionTopLeftToBottomRight();
        formDeadEnds();
    }
    
    
    // This controls how twisty the solution path is.
    // Use 0.0f for maximum straightness.  Use 1.0f for maximum twistyness.
    public float getTwistFactorSolution() { return twistFactorSolution; }
    public void setTwistFactorSolution( float t ) { twistFactorSolution = t; }
    
    // This controls how twisty the dead end paths are.
    // Use 0.0f for maximum straightness.  Use 1.0f for maximum twistyness.
    public float getTwistFactorDeadEnd() { return twistFactorDeadEnd; }
    public void setTwistFactorDeadEnd( float t ) { twistFactorDeadEnd = t; }
    
    
    // Install your own random generator if desired.
    public Random getRandomGenerator() { return random; }
    public void setRandomGenerator( Random generator ) { random = generator; }
    
    
    
    
    //----------------------------------------------------------------------
    //  Internal Support -- random generator
    //----------------------------------------------------------------------
    
    
    // Return a boolean with 50% chance of beith either true or false.
    protected boolean randomBoolean() {
        //		return random.nextInt() < 0;
        return random.nextBoolean();
    }
    
    // Return a random float value from [0..1).
    protected float randomFloat() {
        //		int randomPositiveInt = random.nextInt() & Integer.MAX_VALUE;
        //		return (((float) randomPositiveInt) / ((float) Integer.MAX_VALUE));
        return random.nextFloat();
    }
    
    protected int randomInt( int min, int max ) {
        int randomPositiveInt = random.nextInt() & Integer.MAX_VALUE;
        return randomPositiveInt % (max - min + 1) + min;
    }
    
    
    // Return a boolean.  Probability of it being T is controlled by
    //	the twistFactorSolution variable.
    protected boolean solutionStraightAhead() {
        float randomFloat = randomFloat();
        return randomFloat >= twistFactorSolution;
    }
    
    // Return a boolean.  Probability of it being T is controlled by
    //	the twistFactorDeadEnd variable.
    protected boolean deadEndStraightAhead() {
        float randomFloat = randomFloat();
        return randomFloat >= twistFactorDeadEnd;
    }
    
    // Return a random direction.
    protected int randomDirection() {
        return randomInt( SquareMaze.FIRST_DIRECTION, SquareMaze.LAST_DIRECTION );
    }
    
    
    
    //----------------------------------------------------------------------
    //  Internal Support -- convenience access to the maze.
    //----------------------------------------------------------------------
    
    protected SquareMaze.MazeCell getMazeCell( int row, int col ) {
        return maze.getMazeCell( row, col );
    }
    
    protected int getRows() { return maze.getRows(); }
    protected int getCols() { return maze.getCols(); }
    
    // Syntax sugar for the other class's static functions.
    public static String directionName( int direction ) { return SquareMaze.directionName( direction ); }
    public static int oppositeDirection( int direction ) { return SquareMaze.oppositeDirection( direction ); }
    public static int turnRightDirection( int direction ) { return SquareMaze.turnRightDirection( direction ); }
    public static int turnLeftDirection( int direction ) { return SquareMaze.turnLeftDirection( direction ); }
    
    
    
    //----------------------------------------------------------------------
    //  Internal Support -- properties
    //
    //	The maze generation algorithm needs to attach properties to
    //	 the maze cells.
    //	Add our own virtual properties to a maze cell.
    //----------------------------------------------------------------------
    
    
    //--------------------
    //	Visited
    //  boolean, a cell has been visited while forming a path.
    //--------------------
    
    public static final String PROP_HAS_BEEN_VISITED = "Visited while forming path";
    
    protected boolean hasBeenVisited( SquareMaze.MazeCell cell ) {
        return cell.getB( PROP_HAS_BEEN_VISITED );
    }
    protected void setHasBeenVisited( SquareMaze.MazeCell cell, boolean visited ) {
        cell.setB( PROP_HAS_BEEN_VISITED, visited );
    }
    
    // Mark every cell as having NOT been visited.
    public void resetVisitedFlags() {
        for( int row = 0; row < getRows(); ++row ) {
            for( int col = 0; col < getCols(); ++col ) {
                SquareMaze.MazeCell cell = getMazeCell( row, col );
                setHasBeenVisited( cell, false );
            }
        }
    }
    
    
    //--------------------
    //	Solution Entry Direction
    //  int, the direction that the solution path enters the current cell.
    //--------------------
    
    public static final String PROP_SOLUTION_ENTRY_DIRECTION = "Solution Path Entry Direction";
    
    protected int getSolutionPathEntryDirection( SquareMaze.MazeCell cell ) {
        return cell.getI( PROP_SOLUTION_ENTRY_DIRECTION );
    }
    protected void setSolutionPathEntryDirection( SquareMaze.MazeCell cell, int direction ) {
        cell.setI( PROP_SOLUTION_ENTRY_DIRECTION, direction );
    }
    
    
    //--------------------
    //	Solution Exit Direction
    //  int, the direction that the solution path exits the current cell.
    //--------------------
    
    public static final String PROP_SOLUTION_EXIT_DIRECTION = "Solution Path Exit Direction";
    
    protected int getSolutionPathExitDirection( SquareMaze.MazeCell cell ) {
        return cell.getI( PROP_SOLUTION_EXIT_DIRECTION );
    }
    protected void setSolutionPathExitDirection( SquareMaze.MazeCell cell, int direction ) {
        cell.setI( PROP_SOLUTION_EXIT_DIRECTION, direction );
    }
    
    
    //--------------------
    //	Is On Dead End Path
    //  boolean, a cell is on the dead end path.
    //--------------------
    
    public static final String PROP_DEAD_END_PATH = "Dead end path";
    
    protected boolean isOnDeadEndPath( SquareMaze.MazeCell cell ) {
        return cell.getB( PROP_DEAD_END_PATH );
    }
    protected void setIsOnDeadEndPath( SquareMaze.MazeCell cell, boolean deadEnd ) {
        cell.setB( PROP_DEAD_END_PATH, deadEnd );
    }
    
    
    //--------------------
    //	Dead End Entry Direction
    //  int, the direction that the dead end path enters the current cell.
    //--------------------
    
    public static final String PROP_DEAD_END_ENTRY_DIRECTION = "Solution Path Entry Direction";
    
    protected int getDeadEndPathEntryDirection( SquareMaze.MazeCell cell ) {
        return cell.getI( PROP_DEAD_END_ENTRY_DIRECTION );
    }
    protected void setDeadEndPathEntryDirection( SquareMaze.MazeCell cell, int direction ) {
        cell.setI( PROP_DEAD_END_ENTRY_DIRECTION, direction );
    }
    
    
    //--------------------
    //	Dead End Exit Direction
    //  int, the direction that the dead end path exits the current cell.
    //--------------------
    
    public static final String PROP_DEAD_ENDEXIT_DIRECTION = "Solution Path Exit Direction";
    
    protected int getDeadEndPathExitDirection( SquareMaze.MazeCell cell ) {
        return cell.getI( PROP_DEAD_ENDEXIT_DIRECTION );
    }
    protected void setDeadEndPathExitDirection( SquareMaze.MazeCell cell, int direction ) {
        cell.setI( PROP_DEAD_ENDEXIT_DIRECTION, direction );
    }
    
    
    
    
    //----------------------------------------------------------------------
    //  Internal Support -- build a solution path
    //----------------------------------------------------------------------
    
    protected void formSolutionTopLeftToBottomRight() {
        // Solution from top left to bottom right.
        
        // Start at top left.
        SquareMaze.MazeCell startCell = maze.getMazeCell( 0, 0 );
        // This cell is part of the solution path.
        startCell.setIsOnSolutionPath( true );
        if( randomBoolean() ) {
            // Knock down top wall of start cell.
            startCell.setWall( SquareMaze.DIRECTION_TOP, false );
            // The solution path entered the start cell from the top.
            setSolutionPathEntryDirection( startCell, SquareMaze.DIRECTION_TOP );
        } else {
            // Knock down left wall of start cell.
            startCell.setWall( SquareMaze.DIRECTION_LEFT, false );
            // The solution path entered the start cell from the left.
            setSolutionPathEntryDirection( startCell, SquareMaze.DIRECTION_LEFT );
        }
        
        // End at bottom right.
        SquareMaze.MazeCell endCell = getMazeCell( getRows()-1, getCols()-1 );
        // This cell is part of the solution path.
        endCell.setIsOnSolutionPath( true );
        if( randomBoolean() ) {
            // Knock down bottom wall of end cell.
            endCell.setWall( SquareMaze.DIRECTION_BOTTOM, false );
            // The solution path exits the end cell to the bottom.
            setSolutionPathExitDirection( endCell, SquareMaze.DIRECTION_BOTTOM );
        } else {
            // Knock down right wall of end cell.
            endCell.setWall( SquareMaze.DIRECTION_RIGHT, false );
            // The solution path exits the end cell to the right.
            setSolutionPathExitDirection( endCell, SquareMaze.DIRECTION_RIGHT );
        }
        
        
        formSolution( startCell, endCell );
    }
    
    
    protected void formSolution( SquareMaze.MazeCell startCell, SquareMaze.MazeCell endCell ) {
        SquareMaze.MazeCell nextCell = startCell;
        while( nextCell != endCell ) {
            // Move to the next cell.
            SquareMaze.MazeCell currCell = nextCell;
            
            // Mark currennt cell as having been visited.
            setHasBeenVisited( currCell, true );
            
            // This cell is part of the solution path.
            currCell.setIsOnSolutionPath( true );
            
            // Which direction did we enter the current cell?
            int pathEntryDirection = getSolutionPathEntryDirection( currCell );
            
            // The current cell has three possible exits.
            //
            // The straightAhead direction, is opposite of the direction we entered the cell from.
            int straightAheadDirection = oppositeDirection( pathEntryDirection );
            //
            // There are two alternate turn directions, a right turn or a left turn.
            // Randomize the order of the two turns so we have an equal probability of turning either way.
            int altTurnDirection1, altTurnDirection2;
            if( randomBoolean() ) {
                altTurnDirection1 = turnRightDirection( pathEntryDirection );
                altTurnDirection2 = turnLeftDirection( pathEntryDirection );
            } else {
                altTurnDirection2 = turnRightDirection( pathEntryDirection );
                altTurnDirection1 = turnLeftDirection( pathEntryDirection );
            }
            
            
            // Now form a list of three directions to explore.
            int d1, d2, d3;
            if( solutionStraightAhead() ) {
                // straight ahead is first choice
                d1 = straightAheadDirection;
                // The randomly selected right or left turn are the last choices.
                d2 = altTurnDirection1;
                d3 = altTurnDirection2;
            } else {
                // The randomly selected right or left turn are the first choices.
                d1 = altTurnDirection1;
                d2 = altTurnDirection2;
                // straight ahead is last choice
                d3 = straightAheadDirection;
            }
            
            boolean canConnectCells;
            
            // Explore one of the possible exit directions.
            nextCell = currCell.getNeighboringCell( d1 );
            if( nextCell != null ) {
                if( canConnectSolutionCell( nextCell ) ) {
                    connectTwoSolutionCells( currCell, nextCell, d1 );
                    continue;
                }
            }
            
            // Explore one of the possible exit directions.
            nextCell = currCell.getNeighboringCell( d2 );
            if( nextCell != null ) {
                if( canConnectSolutionCell( nextCell ) ) {
                    connectTwoSolutionCells( currCell, nextCell, d2 );
                    continue;
                }
            }
            
            // Explore one of the possible exit directions.
            nextCell = currCell.getNeighboringCell( d3 );
            if( nextCell != null ) {
                if( canConnectSolutionCell( nextCell ) ) {
                    connectTwoSolutionCells( currCell, nextCell, d3 );
                    continue;
                }
            }
            
            
            // Backtrack -- we could not exit the current cell by any of its possible exits.
            SquareMaze.MazeCell prevCell = currCell.getNeighboringCell( pathEntryDirection );
            if( DEBUG ) System.out.println( "Backtracking solution path from " + currCell + " to " + prevCell );
            nextCell = prevCell;
            // This cell is NOT part of the solution path.
            currCell.setIsOnSolutionPath( false );
            // Re-erect the wall we knocked down to get into this cell.
            currCell.setWall( pathEntryDirection, true );
        } // while
    }
    
    
    protected boolean canConnectSolutionCell( SquareMaze.MazeCell cell ) {
        // If the next cell has not been visited,
        //  and it is not otherwise excluded from being part of the maze,
        //  then we can connect it to the path.
        
        // If the next cell has already been visited while forming this path, then we can NOT connect to it.
        if( hasBeenVisited( cell ) ) return false;
        
        // If the next cell is not excluded from the maze.
        if( ! cell.excludedFromMaze() ) return true;
        
        return false;
    }
    
    
    // Connect two adjacent cells in the solution path.
    protected void connectTwoSolutionCells( SquareMaze.MazeCell currCell, SquareMaze.MazeCell nextCell, int direction ) {
        if( DEBUG ) System.out.println( "Connecting solution path from " + currCell + " going " + directionName( direction ) + " to " + nextCell );
        
        setSolutionPathExitDirection( currCell, direction );
        
        setSolutionPathEntryDirection( nextCell, oppositeDirection( direction ) );
        
        // Knock down the wall between the two cells.
        currCell.setWall( direction, false );
    }
    
    
    
    //----------------------------------------------------------------------
    //  Internal Support -- build dead end paths
    //----------------------------------------------------------------------
    
    // Return an unreachable cell at random.
    // This particular implementation guarantees a bounded runtime.
    protected SquareMaze.MazeCell findRandomUnreachableCell() {
        int maxRandom = Integer.MIN_VALUE;
        SquareMaze.MazeCell unreachableCell = null;
        
        // Try each cell.
        for( int row = 0; row < getRows(); ++row ) {
            for( int col = 0; col < getCols(); ++col ) {
                SquareMaze.MazeCell cell = getMazeCell( row, col );
                
                // This cell can be reached from the maze entrance or exit if
                //  it is on either the solution path, or already on one of the
                //  dead end paths.
                boolean reachable = isReachable( cell );
                
                if( ! reachable ) {
                    // This cell is not reachable, so is a potential candidate to return.
                    // With each unreachable cell we visit, the odds decrease that we will return it.
                    
                    // Get a random integer, that is definitely greater than Integer.MIN_VALUE.
                    // Thus, the very first unreachable cell we encounter is a candidate to return.
                    int rand = random.nextInt();
                    if( rand == Integer.MIN_VALUE ) ++rand;
                    
                    // If this integer is greater than the highest random number we
                    //  have generated so far...
                    if( rand > maxRandom ) {
                        unreachableCell = cell;
                        maxRandom = rand;
                    }
                }
            }
        }
        
        return unreachableCell;
    }
    
    
    protected void formDeadEnds() {
        if( DEBUG ) System.out.println( "Now connecting dead ends." );
        if( DEBUG ) {
            SquareMazePrinter mazePrinter = new SquareMazePrinter1( maze );
            mazePrinter.printMaze( true );
        }
        
        SquareMaze.MazeCell unreachableCell;
        // Loop for as long as there are still unreachable cells.
        while( (unreachableCell = findRandomUnreachableCell()) != null ) {
            resetVisitedFlags();
            formDeadEnd( unreachableCell );
        }
    }
    
    protected void formDeadEnd( SquareMaze.MazeCell startCell ) {
        if( DEBUG ) System.out.println( "Starting new dead end path from " + startCell );
        SquareMaze.MazeCell nextCell = startCell;
        while( true ) {
            // If we have reached a cell reachable from the maze entrance and exit,
            //  then we're done building this dead end path.
            if( isReachable( nextCell ) ) break; // exit loop
            
            
            // Move to the next cell.
            SquareMaze.MazeCell currCell = nextCell;
            
            // Mark currennt cell as having been visited.
            setHasBeenVisited( currCell, true );
            
            // This cell is part of the dead end path.
            setIsOnDeadEndPath( currCell, true );
            
            // Which direction did we enter the current cell?
            int pathEntryDirection = getDeadEndPathEntryDirection( currCell );
            
            // The current cell has three possible exits.
            //
            // The straightAhead direction, is opposite of the direction we entered the cell from.
            int straightAheadDirection = oppositeDirection( pathEntryDirection );
            //
            // There are two alternate turn directions, a right turn or a left turn.
            // Randomize the order of the two turns so we have an equal probability of turning either way.
            int altTurnDirection1, altTurnDirection2;
            if( randomBoolean() ) {
                altTurnDirection1 = turnRightDirection( pathEntryDirection );
                altTurnDirection2 = turnLeftDirection( pathEntryDirection );
            } else {
                altTurnDirection2 = turnRightDirection( pathEntryDirection );
                altTurnDirection1 = turnLeftDirection( pathEntryDirection );
            }
            
            
            // Now form a list of three directions to explore.
            int d1, d2, d3;
            if( deadEndStraightAhead() ) {
                // straight ahead is first choice
                d1 = straightAheadDirection;
                // The randomly selected right or left turn are the last choices.
                d2 = altTurnDirection1;
                d3 = altTurnDirection2;
            } else {
                // The randomly selected right or left turn are the first choices.
                d1 = altTurnDirection1;
                d2 = altTurnDirection2;
                // straight ahead is last choice
                d3 = straightAheadDirection;
            }
            
            boolean canConnectCells;
            
            // Explore one of the possible exit directions.
            nextCell = currCell.getNeighboringCell( d1 );
            if( nextCell != null ) {
                if( canConnectDeadEndCell( nextCell ) ) {
                    connectTwoDeadEndCells( currCell, nextCell, d1 );
                    continue;
                }
            }
            
            // Explore one of the possible exit directions.
            nextCell = currCell.getNeighboringCell( d2 );
            if( nextCell != null ) {
                if( canConnectDeadEndCell( nextCell ) ) {
                    connectTwoDeadEndCells( currCell, nextCell, d2 );
                    continue;
                }
            }
            
            // Explore one of the possible exit directions.
            nextCell = currCell.getNeighboringCell( d3 );
            if( nextCell != null ) {
                if( canConnectDeadEndCell( nextCell ) ) {
                    connectTwoDeadEndCells( currCell, nextCell, d3 );
                    continue;
                }
            }
            
            
            // Backtrack -- we could not exit the current cell by any of its possible exits.
            SquareMaze.MazeCell prevCell = currCell.getNeighboringCell( pathEntryDirection );
            if( DEBUG ) System.out.println( "Backtracking dead end from " + currCell + " to " + prevCell );
            nextCell = prevCell;
            // This cell is NOT part of the dead end path.
            setIsOnDeadEndPath( currCell, false );
            // Re-erect the wall we knocked down to get into this cell.
            currCell.setWall( pathEntryDirection, true );
        } // while
    }
    
    
    protected boolean canConnectDeadEndCell( SquareMaze.MazeCell cell ) {
        // If the next cell is part of the solution path, then this dead end can connect to it.
        if( cell.isOnSolutionPath() ) return true;
        
        // If the next cell has already been visited while forming this path, then we can NOT connect to it.
        if( hasBeenVisited( cell ) ) return false;
        
        // If the next cell is part of the dead end path that has not been visited while forming this path.
        if( isOnDeadEndPath( cell ) ) return true;
        
        // If the next cell is not excluded from the maze.
        if( ! cell.excludedFromMaze() ) return true;
        
        return false;
    }
    
    
    protected boolean isReachable( SquareMaze.MazeCell cell ) {
        // A cell is reachable from the entrance or exit of the maze if it is on the solution path.
        if( cell.isOnSolutionPath() ) {
            if( DEBUG ) System.out.println( "End Path at " + cell + " because it is on solution path." );
            return true;
        }
        
        // If the next cell has already been visited while forming this path, then we can NOT connect to it.
        if( hasBeenVisited( cell ) ) return false;
        
        // A cell is reachable if it is already on a dead end path.
        if( isOnDeadEndPath( cell ) ) {
            if( DEBUG ) System.out.println( "End Path at " + cell + " because it is on dead end path." );
            return true;
        }
        
        return false;
    }
    
    
    // Connect two adjacent cells in the dead end path.
    protected void connectTwoDeadEndCells( SquareMaze.MazeCell currCell, SquareMaze.MazeCell nextCell, int direction ) {
        if( DEBUG ) System.out.println( "Connecting dead end path from " + currCell + " going " + directionName( direction ) + " to " + nextCell );
        
        setDeadEndPathExitDirection( currCell, direction );
        
        setDeadEndPathEntryDirection( nextCell, oppositeDirection( direction ) );
        
        // Knock down the wall between the two cells.
        currCell.setWall( direction, false );
    }
    
    
} // class Maze


